import 'package:dio/dio.dart';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:ready_grocery/models/eCommerce/common/common_response.dart';
import 'package:ready_grocery/models/eCommerce/common/product_filter_model.dart';
import 'package:ready_grocery/models/eCommerce/product/product.dart';
import 'package:ready_grocery/models/eCommerce/product/product_details.dart'
    as product_details;
import 'package:ready_grocery/services/eCommerce/product_service/product_service.dart';
import 'package:ready_grocery/utils/request_handler.dart';

import '../../../models/eCommerce/product/filter.dart';

final selectedColorPriceProvider = StateProvider<double>((ref) => 0);
final selectedSizePriceProvider = StateProvider<double>((ref) => 0);

final productControllerProvider =
    StateNotifierProvider<ProductController, bool>(
        (ref) => ProductController(ref));

class ProductController extends StateNotifier<bool> {
  final Ref ref;
  ProductController(this.ref) : super(false);

  int? _total;
  int? get total => _total;

  List<Product> _products = [];
  List<Product> get products => _products;

  List<Product> _favoriteProducts = [];
  List<Product> get favoriteProducts => _favoriteProducts;

  Filters? _filter;
  Filters? get filter => _filter;
  Future<void> getCategoryWiseProducts({
    required ProductFilterModel productFilterModel,
    required bool isPagination,
  }) async {
    try {
      state = true;
      final response = await ref
          .read(productServiceProvider)
          .getCategoryWiseProducts(productFilterModel: productFilterModel);
      _total = response.data['data']['total'];
      List<dynamic> productData = response.data['data']['products'];
      _filter = Filters.fromMap(response.data['data']['filters']);
      if (isPagination) {
        _products.addAll(
            productData.map((product) => Product.fromMap(product)).toList());
      } else {
        _products =
            productData.map((product) => Product.fromMap(product)).toList();
      }

      state = false;
    } catch (error) {
      debugPrint(error.toString());
      state = false;
      rethrow;
    }
  }

  Future<CommonResponse> favoriteProductAddRemove({
    required int productId,
  }) async {
    try {
      final response = await ref
          .read(productServiceProvider)
          .favoriteProductAddRemove(productId: productId);
      return CommonResponse(isSuccess: true, message: response.data['message']);
    } catch (error) {
      debugPrint(error.toString());
      return CommonResponse(isSuccess: false, message: error.toString());
    }
  }

  Future<CommonResponse> getFavoriteProducts() async {
    try {
      state = true;
      final response =
          await ref.read(productServiceProvider).getFavoriteProducts();
      List<dynamic> favoriteProductsData = response.data['data']['products'];
      _favoriteProducts = favoriteProductsData
          .map((product) => Product.fromMap(product))
          .toList();
      state = false;
      return CommonResponse(isSuccess: true, message: response.data['message']);
    } catch (error) {
      debugPrint(error.toString());
      state = false;
      return CommonResponse(isSuccess: false, message: error.toString());
    }
  }
}

final productDetailsControllerProvider = StateNotifierProvider.family
    .autoDispose<ProductDetailsController,
        AsyncValue<product_details.ProductDetails>, int>((ref, productId) {
  final controller = ProductDetailsController(ref);
  controller.getProductDetails(productId: productId);
  return controller;
});

class ProductDetailsController
    extends StateNotifier<AsyncValue<product_details.ProductDetails>> {
  final Ref ref;
  ProductDetailsController(this.ref) : super(const AsyncLoading());

  Future<void> getProductDetails({required int productId}) async {
    try {
      final response = await ref
          .read(productServiceProvider)
          .getProductDetails(productId: productId);
      final productData = response.data['data'];
      state = AsyncData(product_details.ProductDetails.fromMap(productData));
    } catch (error, stackTrace) {
      debugPrint(error.toString());
      if (mounted) {
        state = AsyncError(
            error is DioException ? ApiInterceptors.handleError(error) : error,
            stackTrace);
      }

      // rethrow;
    }
  }

  // make fav un fav for just for you product
  Future<void> toggleFavourite(Product product) async {
    final previousState = state;
    try {
      state = state.whenData((productDetails) {
        final update = productDetails.relatedProducts.map((p) {
          if (p.id == product.id) {
            return p.copyWith(isFavorite: !p.isFavorite);
          }
          return p;
        }).toList();
        return productDetails.copyWith(relatedProducts: update);
      });
      await ref
          .read(productControllerProvider.notifier)
          .favoriteProductAddRemove(
            productId: product.id,
          );
    } catch (e) {
      state = previousState;
    }
  }
}
